#!/usr/bin/env python3
import os
import json
import sys
import time
import os
import shutil
from kafka import KafkaConsumer

MAX_RETRIES = 5         # How many times to retry connecting to Kafka
RETRY_DELAY = 10        # Seconds to wait between retries

def fetch_env_urls_from_hydra(kafka_endpoint_plain):
    if not kafka_endpoint_plain:
        raise RuntimeError("Missing kafka_endpoint_plain")

    topic = 'hydra'
    try:
        consumer = KafkaConsumer(
            topic,
            bootstrap_servers=[kafka_endpoint_plain],
            auto_offset_reset='earliest',
            enable_auto_commit=False,
            consumer_timeout_ms=5000,
            value_deserializer=lambda m: json.loads(m.decode('utf-8'))
        )
    except Exception as e:
        raise RuntimeError(f"Failed to create Kafka consumer: {e}")

    es_url = None
    wazuh_ctrl_url = None
    tenant_name = None
    key_name = None
    node_ip = None

    try:
        for msg in consumer:
            payload = msg.value
            es_url = payload.get('es', {}).get('url')
            wazuh_ctrl_url = payload.get('wazuh_ctrl', {}).get('url')
            tenant_name = payload.get('tenant_name', {}).get('url')
            key_name = payload.get('key_name', {}).get('url')
            node_ip = payload.get('node_ip', {}).get('url')
            if es_url and wazuh_ctrl_url and tenant_name and key_name and node_ip:
                break
        else:
            raise RuntimeError(f"No valid message on topic '{topic}'")
    finally:
        consumer.close()

    if not es_url:
        raise RuntimeError("ES_URL missing in message")
    if not wazuh_ctrl_url:
        raise RuntimeError("WAZUH_CTRL_URL missing in message")
    if not tenant_name:
        raise RuntimeError("TENANT_NAME missing in message")
    if not key_name:
        raise RuntimeError("KEY_NAME missing in message")
    if not node_ip:
        raise RuntimeError("NODE_IP missing in message")

    return {'es_url': es_url, 'wazuh_ctrl_url': wazuh_ctrl_url, 'tenant_name': tenant_name, 'key_name': key_name, 'node_ip': node_ip}

def write_results_to_common_env(result, env_file="/etc/siemonster/wazuh.env"):
    backup_file = env_file + ".bak"
    existing_keys = set()
    if os.path.exists(env_file):
        with open(env_file, 'r') as f:
            for line in f:
                stripped = line.strip()
                if '=' in stripped:
                    key, _ = stripped.split('=', 1)
                    existing_keys.add(key.strip().upper())
    to_append = []
    for key, value in result.items():
        key_upper = key.upper()
        if key_upper not in existing_keys:
            to_append.append(f"{key_upper}={value}")
    if not to_append:
        print("No new entries to append — all keys already exist.")
        return
    try:
        if os.path.exists(env_file):
            shutil.copy2(env_file, backup_file)
            print(f"Backup created: {backup_file}")
    except Exception as e:
        print(f"ERROR: Failed to create backup: {e}")
        return
    try:
        with open(env_file, 'a') as f:
            for line in to_append:
                f.write(line + '\n')
        print(f"Appended {len(to_append)} new entries to {env_file}")
    except Exception as e:
        print(f"ERROR: Failed to write to {env_file}: {e}")

def main():
    kafka_endpoint_plain = os.getenv('KAFKA_ENDPOINT_PLAIN')
    if not kafka_endpoint_plain:
        print("Error: KAFKA_ENDPOINT_PLAIN environment variable not set.", file=sys.stderr)
        sys.exit(1)

    for attempt in range(1, MAX_RETRIES + 1):
        try:
            result = fetch_env_urls_from_hydra(kafka_endpoint_plain)
        #    print(f"Fetched environment variables: {result}")
            write_results_to_common_env(result)

            return 0  # success
        except Exception as e:
            print(f"Attempt {attempt}: {e}", file=sys.stderr)
            if attempt == MAX_RETRIES:
                print("Max retries reached. Exiting.", file=sys.stderr)
                sys.exit(2)
            time.sleep(RETRY_DELAY)

if __name__ == "__main__":
    main()
