#!/usr/bin/env python
from crhelper import CfnResource

import io
import boto3
import jinja2
import logging
import os

from j2_custom_filters import  wrap_string, b64encode, insert_b64encode


logger = logging.getLogger(__name__)
# Initialise the helper, all inputs are optional, this example shows the defaults
helper = CfnResource(json_logging=False, log_level='DEBUG', boto_level='CRITICAL')
LAMBDA_TASK_ROOT = os.environ.get('LAMBDA_TASK_ROOT', '/')


def get_properties(event):
    allowed_params = [
        'UUIDToken',
        'K8sDomain',
        'ServerIpAddress',
        'KafkaEndpoint',
        'KafkaSecurityProtocol',
        'DisabledServices',
        'BucketName',
        'KeyName',
        'WazuhClusterName', 'WazuhNodeName', 'WazuhNodeType', 'WazuhNodeIp', 'WazuhKey', 'WazuhDisabled',
    ]
    return {k: v for k, v in event['ResourceProperties'].items() if k in allowed_params}


def render_user_data_template(template_dir=LAMBDA_TASK_ROOT, **envs):
    TEMPLATE_USER_DATA = 'files/linux/aws-linux/init.sh.j2'
    for tp in [template_dir, '.', '..']:
        if os.path.exists(os.path.join(tp,TEMPLATE_USER_DATA)):
            template_dir = tp

    templateLoader = jinja2.FileSystemLoader(template_dir)
    templateEnv = jinja2.Environment(loader=templateLoader)
    templateEnv.filters['wrap_string'] = wrap_string
    templateEnv.filters['b64encode'] = b64encode
    templateEnv.filters['insert_b64encode'] = insert_b64encode
    template = templateEnv.get_template(TEMPLATE_USER_DATA)
    userdata = template.render(**envs)
    return userdata


def upload_to_s3(user_data: str, bucket_name, key_name):
    user_data_file_object = io.BytesIO(user_data.encode('utf-8'))
    s3_client = boto3.client('s3')
    s3_client.upload_fileobj(user_data_file_object, bucket_name, key_name)


def delete_from_s3(bucket_name, key_name):
    s3_resource = boto3.resource('s3')
    s3_resource.Object(bucket_name, key_name).delete()


@helper.create
def create(event, context):
    logger.info("Got Create")
    properties = get_properties(event)

    envs = {}
    envs['ENVIRONMENT'] = 'aws'
    envs['K8S_DOMAIN'] = properties['K8sDomain']
    envs['KAFKA_ENDPOINT'] = properties['KafkaEndpoint']
    envs['KAFKA_SECURITY_PROTOCOL'] = properties.get("KafkaSecurityProtocol", "PLAINTEXT")
    envs['UUID_TOKEN'] = properties['UUIDToken']

    envs['WAZUH_CLUSTER_NAME'] = properties['WazuhClusterName']
    envs['WAZUH_NODE_NAME'] = properties['WazuhNodeName']
    envs['WAZUH_NODE_TYPE'] = properties['WazuhNodeType']
    envs['WAZUH_NODE_IP'] = properties['WazuhNodeIp']
    envs['WAZUH_KEY'] = properties['WazuhKey']
    envs['WAZUH_DISABLED'] = properties['WazuhDisabled']

    if properties.get('ServerIpAddress', None):
        envs['SERVER_IP_ADDRESS'] = properties['ServerIpAddress']
    if properties.get('DisabledServices', None):
        envs['DISABLE_SERVICES'] = properties.get('DisabledServices', [])

    rendered = render_user_data_template(**envs)
    upload_to_s3(rendered,properties['BucketName'],properties['KeyName'])

    helper.Data.update({
        "Result": "Created",
        "KeyName": properties['KeyName'],
        "BucketName": properties['BucketName']
    })


@helper.delete
def delete(event, context):
    logger.info("Got Delete")
    properties = get_properties(event)
    delete_from_s3(properties['BucketName'],properties['KeyName'])

    helper.Data.update({
        "Result": "Deleted"
    })


def handler(event, context):
    helper(event, context)


if __name__ == '__main__':
    logger.info('Local tests.')

    logging.info('Creating config file')
    create({'ResourceProperties': {'BucketName': 'cfn-test-bucket-001', 'KeyName': 'init.sh'}}, {})

    logging.info('Deleting config file')
    delete({'ResourceProperties': {'BucketName': 'cfn-test-bucket-001', 'KeyName': 'init.sh'}}, {})