from __future__ import print_function

import os
import re
import sys
import requests
import dns.resolver
from requests.packages.urllib3.exceptions import InsecureRequestWarning
from urllib.parse import urljoin, urlparse

# Disable the InsecureRequestWarning
requests.packages.urllib3.disable_warnings(InsecureRequestWarning)

# Python 3.12 Compatibility
def strtobool(query: str) -> bool:
    if query.lower() in ["y", "yes", "t", "true", "on", "1"]:
        return True
    elif query.lower() in ["n", "no", "f", "false", "off", "0"]:
        return False
    else:
        raise ValueError(f"String value {query} cannot be converted to bool")

# Read configuration from environment
OVPN_URL = os.getenv("OVPN_URL")
OVPN_LOGIN = os.getenv("OVPN_LOGIN")
OVPN_PASSWORD = os.getenv("OVPN_PASSWORD")
OVPN_CERT_NAME = os.getenv("OVPN_CERT_NAME")
OVPN_SSL_VERIFY = strtobool(os.getenv("OVPN_SSL_VERIFY", "false"))

if not all([OVPN_URL, OVPN_LOGIN, OVPN_PASSWORD, OVPN_CERT_NAME]):
    print("ERROR: Missing one of OVPN_URL, OVPN_LOGIN, OVPN_PASSWORD or OVPN_CERT_NAME", file=sys.stderr)
    sys.exit(1)

session = requests.Session()

response = session.get(urljoin(OVPN_URL, '/login'), verify=OVPN_SSL_VERIFY)
response.raise_for_status()

m = re.search(r"""(?<=name="_xsrf" value=")(?P<xsrf_token>[^"]*)""", response.text, flags=re.MULTILINE)
if not m:
    raise RuntimeError("XSRF not found")

xsrf_token = m.group('xsrf_token')

response = session.post(
    urljoin(OVPN_URL, '/login'),
    data={
        'login': OVPN_LOGIN,
        'password': OVPN_PASSWORD,
        '_xsrf': xsrf_token,
    },
    verify=OVPN_SSL_VERIFY
)
response.raise_for_status()
if response.url != urljoin(OVPN_URL, '/'):
    raise RuntimeError("Login failed")


response = session.get(
    urljoin(OVPN_URL, '/certificates'),
    verify=OVPN_SSL_VERIFY
)
response.raise_for_status()
m = re.search(r"""(?<=<a href="/certificates/single-config/%s">)(?P<ovpn_file>[^<]*)""" % OVPN_CERT_NAME, response.text, flags=re.MULTILINE)
if not m:
    print("Certificate %s not found issuing a new one." % OVPN_CERT_NAME)
    response = session.post(
        urljoin(OVPN_URL, '/certificates'),
        data={
            'Name': OVPN_CERT_NAME,
        },
        verify=OVPN_SSL_VERIFY
    )
    response.raise_for_status()


response = session.get(
    urljoin(OVPN_URL, '/certificates/single-config/%s' % OVPN_CERT_NAME),
    verify=OVPN_SSL_VERIFY
)
response.raise_for_status()

with open('files/connection.ovpn', 'w') as fn:
    print(response.text, file=fn)


# --- Kafka environment discovery and persistence ---

def default_k8s_domain():
    """
    Derive the Kubernetes DNS domain from the OVPN URL.
    e.g. vpn.cluster.example.com -> cluster.example.com
    """
    try:
        parsed = urlparse(OVPN_URL)
        return '.'.join(parsed.hostname.split('.')[1:])
    except Exception:
        return None

def kafka_srv_discovery(domain):
    """
    Yield (host, port, is_tls) tuples from Kafka SRV records.
    """
    resolver = dns.resolver.Resolver()
    kafka_prioritized = [
        ("_kafka-tls._tcp", True),
        ("_kafka._tcp", False),
    ]
    for srv, is_tls in kafka_prioritized:
        try:
            # Try the modern API, fallback if unavailable
            try:
                answers = resolver.resolve(f"{srv}.{domain}", "SRV")
            except AttributeError:
                answers = resolver.query(f"{srv}.{domain}", "SRV")

            for r in sorted(answers, key=lambda x: x.priority):
                yield str(r.target).rstrip("."), r.port, is_tls
            # Once we got a working record set, stop trying further SRV names
            break

        except (dns.resolver.NoAnswer, dns.resolver.NXDOMAIN) as e:
            print(f"WARNING: SRV lookup {srv}.{domain} failed: {e}", file=sys.stderr)
            continue

def write_kafka_envs(env_file='wazuh.env'):
    """
    Discover Kafka endpoints and append to the given env file:
      KAFKA_ENDPOINT, KAFKA_ENDPOINT_PLAIN, KAFKA_SECURITY_PROTOCOL
    """
    domain = default_k8s_domain()
    if not domain:
        print("WARNING: Cannot derive k8s domain from OVPN_URL", file=sys.stderr)
        return

    brokers = list(kafka_srv_discovery(domain))
    if not brokers:
        print("WARNING: No Kafka SRV records found for domain:", domain, file=sys.stderr)
        return

    is_tls = all(tls for _, _, tls in brokers)
    kafka_endpoint = ','.join(f"{h}:{p}" for h, p, _ in brokers)
    first = brokers[0][0]
    kafka_endpoint_plain = f"{first}:9092"
    kafka_security_protocol = 'SSL' if is_tls else 'PLAINTEXT'

    # Write to env file (uppercase keys)
    try:
        env_vars = [
            ('KAFKA_ENDPOINT', kafka_endpoint),
            ('KAFKA_ENDPOINT_PLAIN', kafka_endpoint_plain),
            ('KAFKA_SECURITY_PROTOCOL', kafka_security_protocol),
        ]
        with open(env_file, 'w') as f:
            for key, val in env_vars:
                f.write(f"{key}={val}\n")

        print(f"Overwrote Kafka settings to {env_file}")
    except Exception as e:
        print(f"ERROR: Failed writing Kafka envs: {e}", file=sys.stderr)

# Execute writing of Kafka environment variables
write_kafka_envs()
